<?php
/**
 * Plugin Name: KK Product Lineage (Female × Male)
 * Description: Adds "Female Parent" and "Male Parent" fields to WooCommerce products and shows lineage on product pages and archives.
 * Version: 1.0.0
 * Author: CH Consulting Group
 */

if (!defined('ABSPATH')) exit;

if (!function_exists('kkpl_wc_ready')) {
  function kkpl_wc_ready() { return function_exists('WC') && WC(); }
}

/* ======================
 * 1) Admin: Add "Strain Lineage" tab
 * ====================== */
if (!function_exists('kkpl_add_strain_lineage_product_tab')) {
  function kkpl_add_strain_lineage_product_tab($tabs) {
    $tabs['kkpl_strain_lineage'] = array(
      'label'    => __('Strain Lineage', 'kkpl'),
      'target'   => 'kkpl_strain_lineage_data',
      'class'    => array('show_if_simple','show_if_variable','show_if_external','show_if_grouped'),
      'priority' => 70,
    );
    return $tabs;
  }
  add_filter('woocommerce_product_data_tabs', 'kkpl_add_strain_lineage_product_tab', 99);
}

/* ======================
 * 2) Admin: Panel with inputs
 * ====================== */
if (!function_exists('kkpl_strain_lineage_product_panel')) {
  function kkpl_strain_lineage_product_panel() {
    global $post;
    ?>
    <div id="kkpl_strain_lineage_data" class="panel woocommerce_options_panel">
      <div class="options_group">
        <?php
        $female = get_post_meta($post->ID, '_kkpl_strain_mother', true);
        $male   = get_post_meta($post->ID, '_kkpl_strain_father', true);

        woocommerce_wp_text_input(array(
          'id'          => '_kkpl_strain_mother',
          'label'       => __('Female Parent', 'kkpl'),
          'desc_tip'    => true,
          'description' => __('Enter the female (seed-bearing) parent strain.', 'kkpl'),
          'value'       => $female,
        ));

        woocommerce_wp_text_input(array(
          'id'          => '_kkpl_strain_father',
          'label'       => __('Male Parent', 'kkpl'),
          'desc_tip'    => true,
          'description' => __('Enter the male (pollen donor) parent strain.', 'kkpl'),
          'value'       => $male,
        ));
        ?>
      </div>
    </div>
    <?php
  }
  add_action('woocommerce_product_data_panels', 'kkpl_strain_lineage_product_panel');
}

/* ======================
 * 3) Admin: Save fields
 * ====================== */
if (!function_exists('kkpl_save_strain_lineage_fields')) {
  function kkpl_save_strain_lineage_fields($product) {
    if (isset($_POST['_kkpl_strain_mother'])) {
      $product->update_meta_data('_kkpl_strain_mother', sanitize_text_field(wp_unslash($_POST['_kkpl_strain_mother'])));
    }
    if (isset($_POST['_kkpl_strain_father'])) {
      $product->update_meta_data('_kkpl_strain_father', sanitize_text_field(wp_unslash($_POST['_kkpl_strain_father'])));
    }
  }
  add_action('woocommerce_admin_process_product_object', 'kkpl_save_strain_lineage_fields');
}

/* ======================
 * 4) Frontend: Under product title
 * ====================== */
if (!function_exists('kkpl_output_lineage_under_title')) {
  function kkpl_output_lineage_under_title() {
    if (!function_exists('is_product') || !is_product()) return;

    $product_id = get_the_ID();
    $female = trim((string) get_post_meta($product_id, '_kkpl_strain_mother', true));
    $male   = trim((string) get_post_meta($product_id, '_kkpl_strain_father', true));
    if ($female === '' && $male === '') return;

    $parts = array_filter([$female ?: '', $male ?: '']);
    $line  = implode(' × ', $parts);

    echo '<div class="kkpl-lineage" aria-label="' . esc_attr__('Strain Lineage', 'kkpl') . '">
            <span class="kkpl-lineage__icon" aria-hidden="true">
              <svg width="18" height="18" viewBox="0 0 24 24" fill="currentColor" xmlns="http://www.w3.org/2000/svg">
                <path d="M12 2C11.72 3.77 11 8 7 10c2.5.5 4 1 5 4 .97-3 2.5-3.5 5-4-4-2-4.72-6.23-5-8zM6 12c-.79.5-2.4 1.6-3 3 .6.4 1.9.8 3 1 1.07-.46 2.24-.9 3-2-1.33-.33-2.67-1-3-2zm12 0c-.33 1-1.67 1.67-3 2 .76 1.1 1.93 1.54 3 2 1.1-.2 2.4-.6 3-1-.6-1.4-2.21-2.5-3-3zM10 15.5c-.62.5-1.62.8-2.5 1 1.26.93 2.66 1.32 3.5 1.5-.06.84-.06 2.2-.5 3 .6.1 1.3.1 2 0-.44-.8-.44-2.16-.5-3 .84-.18 2.24-.57 3.5-1.5-.88-.2-1.88-.5-2.5-1-.3.62-.66 1.1-1 1.5-.34-.4-.7-.88-1-1.5z"/>
              </svg>
            </span>
            <span class="kkpl-lineage__text">'. esc_html($line) .'</span>
          </div>';
  }
  // Show early under the title (before price/ratings)
  add_action('woocommerce_single_product_summary', 'kkpl_output_lineage_under_title', 6);
}

/* ======================
 * 5) Frontend: Show on product archives (shop/category/search)
 * ====================== */
if (!function_exists('kkpl_show_lineage_in_loop')) {
  function kkpl_show_lineage_in_loop() {
    global $product;
    if (!$product instanceof WC_Product) return;

    if (!(is_shop() || is_product_taxonomy() || is_post_type_archive('product') || (is_search() && get_post_type() === 'product'))) {
      return;
    }

    $pid    = $product->get_id();
    $female = trim((string) get_post_meta($pid, '_kkpl_strain_mother', true));
    $male   = trim((string) get_post_meta($pid, '_kkpl_strain_father', true));
    if ($female === '' && $male === '') return;

    $parts = array_filter([$female ?: '', $male ?: '']);
    $line  = implode(' × ', $parts);

    echo '<div class="kkpl-lineage kkpl-lineage--loop" aria-label="' . esc_attr__('Strain Lineage', 'kkpl') . '">
            <span class="kkpl-lineage__icon" aria-hidden="true">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor" xmlns="http://www.w3.org/2000/svg">
                <path d="M12 2C11.72 3.77 11 8 7 10c2.5.5 4 1 5 4 .97-3 2.5-3.5 5-4-4-2-4.72-6.23-5-8zM6 12c-.79.5-2.4 1.6-3 3 .6.4 1.9.8 3 1 1.07-.46 2.24-.9 3-2-1.33-.33-2.67-1-3-2zm12 0c-.33 1-1.67 1.67-3 2 .76 1.1 1.93 1.54 3 2 1.1-.2 2.4-.6 3-1-.6-1.4-2.21-2.5-3-3zM10 15.5c-.62.5-1.62.8-2.5 1 1.26.93 2.66 1.32 3.5 1.5-.06.84-.06 2.2-.5 3 .6.1 1.3.1 2 0-.44-.8-.44-2.16-.5-3 .84-.18 2.24-.57 3.5-1.5-.88-.2-1.88-.5-2.5-1-.3.62-.66 1.1-1 1.5-.34-.4-.7-.88-1-1.5z"/>
              </svg>
            </span>
            <span class="kkpl-lineage__text">'. esc_html($line) .'</span>
          </div>';
  }
  add_action('woocommerce_after_shop_loop_item_title', 'kkpl_show_lineage_in_loop', 4);
}

/* ======================
 * 6) Optional Shortcode: [kk_lineage id="123"]
 * ====================== */
if (!function_exists('kkpl_shortcode_lineage')) {
  function kkpl_shortcode_lineage($atts = array()) {
    $atts = shortcode_atts(array('id' => 0), $atts, 'kk_lineage');
    $pid  = absint($atts['id']) ?: get_the_ID();
    if (!$pid) return '';

    $female = trim((string) get_post_meta($pid, '_kkpl_strain_mother', true));
    $male   = trim((string) get_post_meta($pid, '_kkpl_strain_father', true));
    if ($female === '' && $male === '') return '';

    $parts = array_filter([$female ?: '', $male ?: '']);
    $line  = implode(' × ', $parts);

    ob_start();
    echo '<div class="kkpl-lineage" aria-label="' . esc_attr__('Strain Lineage', 'kkpl') . '">
            <span class="kkpl-lineage__icon" aria-hidden="true">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor" xmlns="http://www.w3.org/2000/svg">
                <path d="M12 2C11.72 3.77 11 8 7 10c2.5.5 4 1 5 4 .97-3 2.5-3.5 5-4-4-2-4.72-6.23-5-8zM6 12c-.79.5-2.4 1.6-3 3 .6.4 1.9.8 3 1 1.07-.46 2.24-.9 3-2-1.33-.33-2.67-1-3-2zm12 0c-.33 1-1.67 1.67-3 2 .76 1.1 1.93 1.54 3 2 1.1-.2 2.4-.6 3-1-.6-1.4-2.21-2.5-3-3zM10 15.5c-.62.5-1.62.8-2.5 1 1.26.93 2.66 1.32 3.5 1.5-.06.84-.06 2.2-.5 3 .6.1 1.3.1 2 0-.44-.8-.44-2.16-.5-3 .84-.18 2.24-.57 3.5-1.5-.88-.2-1.88-.5-2.5-1-.3.62-.66 1.1-1 1.5-.34-.4-.7-.88-1-1.5z"/>
              </svg>
            </span>
            <span class="kkpl-lineage__text">'. esc_html($line) .'</span>
          </div>';
    return ob_get_clean();
  }
  add_shortcode('kk_lineage', 'kkpl_shortcode_lineage');
}

/* ======================
 * 7) Styles
 * ====================== */
add_action('wp_head', function () { ?>
  <style>
    .kkpl-lineage{display:flex;align-items:center;gap:.5rem;margin:.25rem 0 .5rem;font-size:15px;line-height:1.4;color:#2c2c2c}
    .kkpl-lineage__icon{display:inline-flex;width:18px;height:18px;color:#3b7d3b}
    .kkpl-lineage__text{font-weight:500;letter-spacing:.2px}
    /* archive/grid tweaks */
    ul.products li.product .kkpl-lineage--loop{margin:.25rem 0 .35rem}
  </style>
<?php });

